// Vim-style keyboard navigation for Freshdesk ticket list
// Phase 1: j/k navigation + preview popup

console.log('🎹 Replymate Vim Navigation loaded');

class VimNavigation {
  constructor() {
    this.enabled = false;
    this.ticketPageEscapeEnabled = false;
    this.ticketPageEscapeHandler = null;
    this.currentIndex = -1;
    this.tickets = [];
    this.previewPopup = null;
    this.replyCache = {}; // Cache for pre-generated replies: {ticketId: {reply, timestamp}}
    this.cacheTimeout = 5 * 60 * 1000; // 5 minutes
    this.isGenerating = {}; // Track which tickets are currently generating
    this.init();
  }

  init() {
    // Only activate on ticket list pages
    if (this.isTicketListPage()) {
      this.enable();
    } else if (this.isTicketPage()) {
      this.enableTicketPageEscape();
    }

    // Watch for navigation to ticket list (SPA routing)
    this.watchForPageChanges();
  }

  isTicketListPage() {
    // Check if we're on the all tickets page
    const url = window.location.href;
    return url.includes('/a/tickets') && !url.match(/\/tickets\/\d+/);
  }

  isTicketPage() {
    // Check if we're on an individual ticket page
    const url = window.location.href;
    return url.match(/\/a\/tickets\/\d+/);
  }

  watchForPageChanges() {
    let lastUrl = location.href;
    new MutationObserver(() => {
      const url = location.href;
      if (url !== lastUrl) {
        lastUrl = url;

        if (this.isTicketListPage()) {
          this.disableTicketPageEscape();
          this.enable();
        } else if (this.isTicketPage()) {
          this.disable();
          this.enableTicketPageEscape();
        } else {
          this.disable();
          this.disableTicketPageEscape();
        }
      }
    }).observe(document.body, { subtree: true, childList: true });
  }

  enable() {
    if (this.enabled) return;

    console.log('✓ Vim navigation enabled');
    this.enabled = true;

    // Check if we just returned from a modal via Esc
    const justReturned = sessionStorage.getItem('replymate-vim-just-returned');
    if (justReturned) {
      sessionStorage.removeItem('replymate-vim-just-returned');
      console.log('🔙 Returned from modal - aggressively blurring search box');
    }

    // Aggressively blur any focused input (e.g., search filter) so keyboard shortcuts work
    // Do this multiple times because Freshdesk may try to refocus
    const blurInputs = () => {
      if (document.activeElement && ['INPUT', 'TEXTAREA'].includes(document.activeElement.tagName)) {
        document.activeElement.blur();
        console.log('🔇 Blurred:', document.activeElement.tagName);
      }
    };

    // Continuous blur watcher - runs every 50ms for 3 seconds (or 5 seconds if just returned)
    const duration = justReturned ? 5000 : 3000;
    const startTime = Date.now();
    const blurInterval = setInterval(() => {
      blurInputs();
      if (Date.now() - startTime > duration) {
        clearInterval(blurInterval);
        console.log('🛑 Stopped continuous blur watcher');
      }
    }, 50);

    // Wait for ticket list to load
    setTimeout(() => {
      this.refreshTicketList();
      this.attachKeyboardListener();
      this.showStatusIndicator();
    }, 500);
  }

  disable() {
    if (!this.enabled) return;

    console.log('✗ Vim navigation disabled');
    this.enabled = false;
    this.currentIndex = -1;
    this.removeHighlight();
    this.hidePreview();
    this.hideStatusIndicator();
  }

  enableTicketPageEscape() {
    if (this.ticketPageEscapeEnabled) return;

    console.log('✓ Ticket page Escape enabled');
    this.ticketPageEscapeEnabled = true;

    // Bind the handler
    this.ticketPageEscapeHandler = this.handleTicketPageEscape.bind(this);
    document.addEventListener('keydown', this.ticketPageEscapeHandler, true); // Use capture phase

    // Show indicator
    this.showTicketPageIndicator();
  }

  disableTicketPageEscape() {
    if (!this.ticketPageEscapeEnabled) return;

    console.log('✗ Ticket page Escape disabled');
    this.ticketPageEscapeEnabled = false;

    // Remove the handler
    if (this.ticketPageEscapeHandler) {
      document.removeEventListener('keydown', this.ticketPageEscapeHandler, true); // Must match capture phase
      this.ticketPageEscapeHandler = null;
    }

    // Hide indicator
    this.hideTicketPageIndicator();
  }

  handleTicketPageEscape(e) {
    // FIRST: Check global flag - this is set by modal handlers
    if (window.replymateModalOpen === true) {
      // Modal is handling keys - do NOT process anything
      return;
    }

    // Ignore if typing in input/textarea
    if (['INPUT', 'TEXTAREA'].includes(e.target.tagName)) return;
    if (e.target.isContentEditable) return;

    if (e.key === 'Escape') {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
      // Go back to ticket list
      window.history.back();
    }
  }

  showTicketPageIndicator() {
    const existing = document.getElementById('replymate-ticket-escape-indicator');
    if (existing) return;

    const indicator = document.createElement('div');
    indicator.id = 'replymate-ticket-escape-indicator';
    indicator.className = 'replymate-vim-indicator';
    indicator.innerHTML = `
      <div class="replymate-vim-indicator-content">
        🎹 Press <strong>Esc</strong> to return to ticket list
      </div>
    `;
    document.body.appendChild(indicator);
  }

  hideTicketPageIndicator() {
    const indicator = document.getElementById('replymate-ticket-escape-indicator');
    if (indicator) {
      indicator.remove();
    }
  }

  refreshTicketList() {
    // Find all ticket cards in the DOM
    // Freshdesk card view: each ticket is a card with a link
    const ticketLinks = document.querySelectorAll('a[href*="/tickets/"]');

    // Get the parent card element for each ticket link
    const cards = Array.from(ticketLinks)
      .map(link => {
        // Find the card container - usually 3-4 levels up from the link
        let card = link;
        for (let i = 0; i < 5; i++) {
          card = card.parentElement;
          if (!card) break;

          // Check if this element looks like a card container
          // Cards typically have multiple child elements (avatar, subject, metadata)
          if (card.children.length >= 3) {
            return card;
          }
        }
        return null;
      })
      .filter((card, index, self) => {
        // Remove nulls and duplicates
        return card !== null && self.indexOf(card) === index;
      });

    if (cards.length > 0) {
      this.tickets = cards;
      console.log(`Found ${this.tickets.length} tickets`);
    } else {
      console.warn('Could not find ticket cards');
    }
  }

  attachKeyboardListener() {
    this.boundHandleKeyPress = this.handleKeyPress.bind(this);
    document.addEventListener('keydown', this.boundHandleKeyPress, true); // Use capture phase
  }

  handleKeyPress(e) {
    // FIRST: Check global flag - this is set by modal handlers
    if (window.replymateModalOpen === true) {
      // Modal is handling keys - do NOT process anything
      return;
    }

    if (!this.enabled) return;

    // Ignore if typing in input/textarea
    if (['INPUT', 'TEXTAREA'].includes(e.target.tagName)) return;
    if (e.target.isContentEditable) return;

    switch(e.key) {
      case 'j':
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        this.selectNext();
        break;
      case 'k':
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        this.selectPrevious();
        break;
      case 'g':
        // Check for 'gg' (go to first)
        if (this.lastKey === 'g') {
          e.preventDefault();
          e.stopPropagation();
          e.stopImmediatePropagation();
          this.selectFirst();
        }
        break;
      case 'G':
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        this.selectLast();
        break;
      case 'Escape':
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        this.hidePreview();
        break;
      case 'i':
        // Generate AI reply for current ticket
        e.preventDefault();
        e.stopPropagation();
        e.stopImmediatePropagation();
        this.generateReplyForCurrentTicket();
        break;
    }

    this.lastKey = e.key;
    setTimeout(() => this.lastKey = null, 500);
  }

  selectNext() {
    if (this.tickets.length === 0) {
      this.refreshTicketList();
      return;
    }

    if (this.currentIndex < this.tickets.length - 1) {
      this.currentIndex++;
      this.selectTicket(this.currentIndex);
    }
  }

  selectPrevious() {
    if (this.tickets.length === 0) {
      this.refreshTicketList();
      return;
    }

    if (this.currentIndex > 0) {
      this.currentIndex--;
      this.selectTicket(this.currentIndex);
    } else if (this.currentIndex === -1 && this.tickets.length > 0) {
      this.currentIndex = 0;
      this.selectTicket(this.currentIndex);
    }
  }

  selectFirst() {
    if (this.tickets.length === 0) return;
    this.currentIndex = 0;
    this.selectTicket(this.currentIndex);
  }

  selectLast() {
    if (this.tickets.length === 0) return;
    this.currentIndex = this.tickets.length - 1;
    this.selectTicket(this.currentIndex);
  }

  selectTicket(index) {
    // Remove previous highlight
    this.removeHighlight();

    // Add highlight to current ticket
    const ticket = this.tickets[index];
    ticket.classList.add('replymate-vim-selected');

    // Scroll into view if needed
    ticket.scrollIntoView({ behavior: 'smooth', block: 'nearest' });

    // Fetch and show preview with full message content
    this.fetchAndShowPreview(ticket);

    // Pre-generate reply for next ticket (look-ahead)
    if (index < this.tickets.length - 1) {
      this.preGenerateReply(this.tickets[index + 1]);
    }
  }

  async fetchAndShowPreview(ticket) {
    // Show initial preview
    this.showPreview(ticket);

    // Extract ticket ID
    const link = ticket.querySelector('a[href*="/tickets/"]');
    if (!link) return;

    const match = link.href.match(/\/tickets\/(\d+)/);
    if (!match) return;

    const ticketId = match[1];

    // Fetch ticket description from Freshdesk API
    try {
      const response = await fetch(`https://robomateco-help.freshdesk.com/api/v2/tickets/${ticketId}`, {
        headers: {
          'Authorization': 'Basic ' + btoa('szMA73aUly74WwQBUrcV:X'),
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const ticketData = await response.json();

        if (ticketData.description_text) {
          // Update preview with the actual message content
          const data = this.extractTicketData(ticket);
          data.preview = ticketData.description_text.trim().substring(0, 500);
          this.renderPreview(data, ticket);
        }
      }
    } catch (error) {
      console.error('Failed to fetch ticket details:', error);
    }
  }

  showPreviewWithMessage(ticket, messageContent) {
    const data = this.extractTicketData(ticket);
    // Strip HTML tags from message if present
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = messageContent;
    const cleanText = tempDiv.textContent || tempDiv.innerText || messageContent;
    data.preview = cleanText.substring(0, 500);

    this.renderPreview(data, ticket);
  }

  removeHighlight() {
    const selected = document.querySelector('.replymate-vim-selected');
    if (selected) {
      selected.classList.remove('replymate-vim-selected');
    }
  }


  showPreview(ticketRow) {
    const data = this.extractTicketData(ticketRow);
    if (!data) return;

    this.renderPreview(data, ticketRow);
  }

  renderPreview(data, ticketRow) {
    // Remove existing preview
    this.hidePreview();

    // Create preview popup
    this.previewPopup = document.createElement('div');
    this.previewPopup.id = 'replymate-vim-preview';
    this.previewPopup.className = 'replymate-vim-preview';
    this.previewPopup.innerHTML = `
      <div class="replymate-vim-preview-header">
        <div class="replymate-vim-preview-title">${this.escapeHtml(data.subject)}</div>
        <div class="replymate-vim-preview-meta">
          <span class="replymate-vim-preview-ticket">Ticket #${data.ticketId}</span>
          <span class="replymate-vim-preview-customer">${this.escapeHtml(data.customer)}</span>
          ${data.status ? `<span class="replymate-vim-preview-status">${this.escapeHtml(data.status)}</span>` : ''}
        </div>
      </div>
      <div class="replymate-vim-preview-body">
        ${this.escapeHtml(data.preview)}
      </div>
      <div class="replymate-vim-preview-footer">
        Press <strong>i</strong> to <em>i</em>nsert AI reply
      </div>
    `;

    document.body.appendChild(this.previewPopup);

    // Position popup
    const ticket = ticketRow || this.tickets[this.currentIndex];
    if (ticket) {
      this.positionPreview(ticket);
    }
  }

  positionPreview(ticketRow) {
    if (!this.previewPopup) return;

    const rect = ticketRow.getBoundingClientRect();
    const previewWidth = 400;
    const previewHeight = this.previewPopup.offsetHeight;

    // Try to position to the right of the row
    let left = rect.right + 20;
    let top = rect.top;

    // If it would go off-screen right, position on left
    if (left + previewWidth > window.innerWidth) {
      left = rect.left - previewWidth - 20;
    }

    // If it would go off-screen left, center it
    if (left < 0) {
      left = (window.innerWidth - previewWidth) / 2;
    }

    // Keep within viewport vertically
    if (top + previewHeight > window.innerHeight) {
      top = window.innerHeight - previewHeight - 20;
    }
    if (top < 0) {
      top = 20;
    }

    this.previewPopup.style.left = `${left}px`;
    this.previewPopup.style.top = `${top}px`;
  }

  hidePreview() {
    if (this.previewPopup) {
      this.previewPopup.remove();
      this.previewPopup = null;
    }
  }

  extractTicketData(ticketRow) {
    // Extract ticket information from the card
    const data = {
      ticketId: '',
      subject: '',
      customer: '',
      status: '',
      preview: ''
    };

    // Extract ticket ID and subject from link
    const link = ticketRow.querySelector('a[href*="/tickets/"]');
    if (link) {
      const match = link.href.match(/\/tickets\/(\d+)/);
      if (match) {
        data.ticketId = match[1];
      }

      // Subject is the link text, extract just the subject part (before #number)
      const fullText = link.textContent.trim();
      const hashIndex = fullText.lastIndexOf('#');
      if (hashIndex > 0) {
        data.subject = fullText.substring(0, hashIndex).trim();
      } else {
        data.subject = fullText;
      }
    }

    // Extract customer name - look for contact links
    const customerLink = ticketRow.querySelector('a[href*="/contacts/"]');
    if (customerLink) {
      data.customer = customerLink.textContent.trim();
    }

    // Extract status - look for "Open", "Closed", "Pending", etc
    const allText = ticketRow.textContent;
    const statusKeywords = ['Open', 'Closed', 'Pending', 'Resolved', 'Waiting'];
    for (const keyword of statusKeywords) {
      if (allText.includes(keyword)) {
        data.status = keyword;
        break;
      }
    }

    // Extract preview - look for expanded message content
    // When expanded, Freshdesk shows: [timestamp header] [message content] [Reply/Add note buttons]

    // Check if ticket is expanded by looking for Reply button
    const replyButton = ticketRow.querySelector('button');
    const isExpanded = replyButton && replyButton.textContent.includes('Reply');

    if (isExpanded) {
      // Ticket is expanded - extract the message content
      // Message is typically between the timestamp and the Reply button
      const textLines = (ticketRow.innerText || ticketRow.textContent || '').split('\n')
        .map(line => line.trim())
        .filter(line => line.length > 0);

      let messageLines = [];
      let foundMessage = false;

      for (const line of textLines) {
        // Skip until we get past the metadata header
        if (line.includes('submitted a new ticket') ||
            line.includes('hours ago') ||
            line.includes('minutes ago') ||
            line.includes('days ago')) {
          foundMessage = true;
          continue;
        }

        // Stop when we hit the Reply/Add note buttons
        if (line === 'Reply' || line === 'Add note') {
          break;
        }

        // Collect message lines
        if (foundMessage && line.length > 0 &&
            !line.includes('Created:') && !line.includes('Closed:') &&
            !line.includes('due in:') && !line.includes('Priority') &&
            line !== 'New' && !statusKeywords.includes(line) &&
            !line.includes('Low') && !line.includes('Medium') && !line.includes('High')) {
          messageLines.push(line);
        }
      }

      if (messageLines.length > 0) {
        data.preview = messageLines.join(' ').substring(0, 500);
      }
    }

    // Fallback if no preview extracted
    if (!data.preview) {
      const metaInfo = [];
      if (data.status) metaInfo.push(data.status);

      const timeMatch = allText.match(/(Created|Closed):\s*([^•\n]+)/);
      if (timeMatch) {
        metaInfo.push(timeMatch[2].trim());
      }

      data.preview = metaInfo.length > 0 ? metaInfo.join(' • ') + ' • Fetching message...' : 'Fetching message...';
    }

    return data;
  }

  showStatusIndicator() {
    // Show a small indicator that vim mode is active
    const existing = document.getElementById('replymate-vim-indicator');
    if (existing) return;

    const indicator = document.createElement('div');
    indicator.id = 'replymate-vim-indicator';
    indicator.className = 'replymate-vim-indicator';

    const logoUrl = chrome.runtime.getURL('icons/robomate-logo.png');

    indicator.innerHTML = `
      <div class="replymate-vim-indicator-content">
        <img src="${logoUrl}" class="replymate-vim-logo" alt="Replymate"> Replymate: <strong>j</strong>/<strong>k</strong> navigate • <strong>gg</strong>/<strong>G</strong> first/last • <strong>i</strong> AI reply
      </div>
    `;
    document.body.appendChild(indicator);
  }

  hideStatusIndicator() {
    const indicator = document.getElementById('replymate-vim-indicator');
    if (indicator) {
      indicator.remove();
    }
  }

  getCachedReply(ticketId) {
    const cached = this.replyCache[ticketId];
    if (!cached) return null;

    // Check if cache is still valid (within 5 minutes)
    const age = Date.now() - cached.timestamp;
    if (age > this.cacheTimeout) {
      delete this.replyCache[ticketId];
      return null;
    }

    return cached.reply;
  }

  async preGenerateReply(ticket) {
    const data = this.extractTicketData(ticket);
    if (!data.ticketId) return;

    // Don't pre-generate if already cached or currently generating
    if (this.getCachedReply(data.ticketId) || this.isGenerating[data.ticketId]) {
      return;
    }

    // Mark as generating
    this.isGenerating[data.ticketId] = true;

    try {
      // Fetch full message content
      const response = await fetch(`https://robomateco-help.freshdesk.com/api/v2/tickets/${data.ticketId}`, {
        headers: {
          'Authorization': 'Basic ' + btoa('szMA73aUly74WwQBUrcV:X'),
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        delete this.isGenerating[data.ticketId];
        return;
      }

      const ticketData = await response.json();
      const customerMessage = ticketData.description_text?.trim();

      if (!customerMessage || customerMessage.length < 20) {
        delete this.isGenerating[data.ticketId];
        return;
      }

      // Check if assistant is available
      if (typeof assistant === 'undefined') {
        delete this.isGenerating[data.ticketId];
        return;
      }

      // Build context and generate reply
      const context = {
        subject: data.subject,
        customerMessage: customerMessage,
        customerName: data.customer,
        customerEmail: '',
        ticketId: data.ticketId
      };

      const reply = await assistant.generateReply(context);

      // Cache the reply
      this.replyCache[data.ticketId] = {
        reply: reply,
        timestamp: Date.now()
      };

      console.log(`✓ Pre-generated reply for ticket #${data.ticketId}`);

    } catch (error) {
      console.error('Pre-generation failed:', error);
    } finally {
      delete this.isGenerating[data.ticketId];
    }
  }

  async generateReplyForCurrentTicket() {
    if (this.currentIndex === -1) return;

    const ticket = this.tickets[this.currentIndex];

    // Extract ticket data
    const data = this.extractTicketData(ticket);

    if (!data.ticketId) {
      alert('Cannot generate reply: missing ticket ID');
      return;
    }

    // Get the ticket link to navigate later
    const link = ticket.querySelector('a[href*="/tickets/"]');
    const ticketUrl = link ? link.href : null;

    // Fetch latest customer message from conversations
    let customerMessage = '';
    try {
      // Fetch all conversations for this ticket
      const response = await fetch(`https://robomateco-help.freshdesk.com/api/v2/tickets/${data.ticketId}/conversations`, {
        headers: {
          'Authorization': 'Basic ' + btoa('szMA73aUly74WwQBUrcV:X'),
          'Content-Type': 'application/json'
        }
      });

      if (response.ok) {
        const conversations = await response.json();

        // Filter to only customer messages (incoming: true) and sort by newest first
        const customerMessages = conversations
          .filter(conv => conv.incoming === true)
          .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));

        if (customerMessages.length > 0) {
          // Get the most recent customer message
          const latestMessage = customerMessages[0];
          customerMessage = (latestMessage.body_text || latestMessage.body || '').trim();
          console.log(`📧 Found latest customer message from ${latestMessage.created_at}`);
        }
      }

      // Fallback: if no conversations found, try to get original description
      if (!customerMessage || customerMessage.length < 20) {
        const ticketResponse = await fetch(`https://robomateco-help.freshdesk.com/api/v2/tickets/${data.ticketId}`, {
          headers: {
            'Authorization': 'Basic ' + btoa('szMA73aUly74WwQBUrcV:X'),
            'Content-Type': 'application/json'
          }
        });

        if (ticketResponse.ok) {
          const ticketData = await ticketResponse.json();
          customerMessage = (ticketData.description_text || '').trim();
          console.log('📝 Using original ticket description as fallback');
        }
      }
    } catch (error) {
      console.error('Failed to fetch customer message:', error);
    }

    if (!customerMessage || customerMessage.length < 20) {
      alert('Cannot generate reply: no customer message found');
      return;
    }

    // Build context object for display
    const context = {
      subject: data.subject,
      customerMessage: customerMessage,
      customerName: data.customer,
      customerEmail: '',
      ticketId: data.ticketId
    };

    // Store return URL (current page)
    const returnUrl = window.location.href;

    // Check cache - if cached, navigate and show result
    const cachedReply = this.getCachedReply(data.ticketId);
    if (cachedReply) {
      console.log(`✓ Using cached reply for ticket #${data.ticketId}`);
      // Store data for modal display on ticket page
      sessionStorage.setItem('replymate-vim-modal-reply', cachedReply);
      sessionStorage.setItem('replymate-vim-modal-context', JSON.stringify(context));
      sessionStorage.setItem('replymate-vim-return-url', returnUrl);
      // Navigate to ticket page
      window.location.href = ticketUrl;
      return;
    }

    // Check if we have the AI assistant instance
    if (typeof assistant === 'undefined') {
      alert('AI assistant not initialized');
      return;
    }

    // Show loading
    this.showAILoading();

    try {
      // Generate reply using the existing assistant
      const reply = await assistant.generateReply(context);

      // Cache the reply
      this.replyCache[data.ticketId] = {
        reply: reply,
        timestamp: Date.now()
      };

      // Hide loading
      this.hideAILoading();

      // Store data for modal display on ticket page
      sessionStorage.setItem('replymate-vim-modal-reply', reply);
      sessionStorage.setItem('replymate-vim-modal-context', JSON.stringify(context));
      sessionStorage.setItem('replymate-vim-return-url', returnUrl);

      // Navigate to ticket page
      window.location.href = ticketUrl;

    } catch (error) {
      this.hideAILoading();
      alert('Error generating reply: ' + error.message);
      console.error('Error:', error);
    }
  }

  showAILoading() {
    const existing = document.getElementById('replymate-ai-loading');
    if (existing) existing.remove();

    const overlay = document.createElement('div');
    overlay.id = 'replymate-ai-loading';
    overlay.className = 'robomate-ai-overlay';
    overlay.innerHTML = `
      <div class="robomate-ai-modal">
        <div class="robomate-ai-header">
          <div class="robomate-ai-title">Replymate</div>
        </div>
        <div class="robomate-ai-body">
          <div class="robomate-ai-loading-text">Analyzing ticket and generating reply...</div>
          <div class="robomate-ai-progress">
            <div class="robomate-ai-progress-bar"></div>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(overlay);
  }

  hideAILoading() {
    const loading = document.getElementById('replymate-ai-loading');
    if (loading) loading.remove();
  }

  showAIResult(reply, ticketUrl, context, returnUrl = null) {
    const existing = document.getElementById('replymate-ai-result');
    if (existing) existing.remove();

    // Use assistant's createInteractiveReply to get dropdowns
    const interactiveHtml = assistant.createInteractiveReply(reply);

    const overlay = document.createElement('div');
    overlay.id = 'replymate-ai-result';
    overlay.className = 'robomate-ai-overlay';
    overlay.innerHTML = `
      <div class="robomate-ai-modal robomate-ai-modal-large">
        <div class="robomate-ai-header">
          <div class="robomate-ai-title">Replymate Suggestion</div>
          <button class="robomate-ai-close" id="robomate-close-btn">×</button>
        </div>
        <div class="robomate-ai-body">
          <div class="robomate-top-row">
            <div class="robomate-customer-message">
              <div class="robomate-message-header">
                <span class="robomate-message-icon">💬</span>
                <span class="robomate-message-from">Message from ${this.escapeHtml(context.customerName || 'Customer')}</span>
              </div>
              <div class="robomate-message-content">${this.escapeHtml(context.customerMessage || 'No message content')}</div>
            </div>
            <!-- COMMENTED OUT: Keyboard navigation compass - can resurrect later
            <div class="robomate-vim-compass">
              <div class="robomate-compass-title">Keyboard Navigation</div>
              <div class="robomate-compass-rose">
                <div class="robomate-compass-key robomate-compass-up">
                  <span class="robomate-key">k</span>
                  <span class="robomate-label">UP</span>
                </div>
                <div class="robomate-compass-center">
                  <div class="robomate-compass-key robomate-compass-left">
                    <span class="robomate-key">h</span>
                    <span class="robomate-label">PREV DROPDOWN</span>
                  </div>
                  <div class="robomate-compass-middle">
                    <span class="robomate-vim-text">VIM</span>
                  </div>
                  <div class="robomate-compass-key robomate-compass-right">
                    <span class="robomate-key">l</span>
                    <span class="robomate-label">NEXT DROPDOWN</span>
                  </div>
                </div>
                <div class="robomate-compass-key robomate-compass-down">
                  <span class="robomate-key">j</span>
                  <span class="robomate-label">DOWN</span>
                </div>
              </div>
              <div class="robomate-compass-actions">
                <span><strong>dd</strong> delete line</span> •
                <span><strong>u</strong> undo</span> •
                <span><strong>c</strong> copy</span> •
                <span><strong>o</strong> open & insert</span> •
                <span><strong>Esc</strong> close</span>
              </div>
            </div>
            END COMMENTED OUT -->
          </div>
          <div class="robomate-our-reply-section">
            <div class="robomate-reply-header">
              <span class="robomate-reply-title">Our Reply</span>
              <button class="robomate-manual-edit-btn" id="robomate-manual-edit-btn" title="Switch to manual editing mode">Manual Edit</button>
              <span class="robomate-reply-help-text">ℹ️ Hover lines to delete</span>
            </div>
            <div class="robomate-ai-reply-preview" id="robomate-reply-preview">${interactiveHtml}</div>
          </div>
          <div class="robomate-feedback-comment-section">
            <textarea
              class="robomate-feedback-comment"
              id="robomate-feedback-comment"
              placeholder="Optional: Add any comments about this generation..."
              rows="2"></textarea>
          </div>
          <div class="robomate-ai-actions">
            <button class="robomate-ai-btn-secondary" id="robomate-copy-btn">📋 Copy to Clipboard</button>
            <button class="robomate-ai-btn-tertiary" id="robomate-discard-btn">✕ Open Ticket (Discard Suggestion)</button>
            <button class="robomate-ai-btn-primary" id="robomate-insert-btn">✓ Open Ticket & Insert Reply</button>
          </div>
        </div>
      </div>
    `;

    document.body.appendChild(overlay);

    // Set global flag to block vim navigation
    window.replymateModalOpen = true;

    // State for sentence line editing
    let selectedLineId = null;
    let lastKeyPressed = null;
    let lastKeyTime = 0;

    // Helper to select a sentence line
    const selectLine = (lineId) => {
      const lines = overlay.querySelectorAll('.robomate-sentence-line');
      lines.forEach(line => line.classList.remove('selected'));
      const targetLine = overlay.querySelector(`.robomate-sentence-line[data-sentence-id="${lineId}"]`);
      if (targetLine) {
        targetLine.classList.add('selected');
        targetLine.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
        selectedLineId = lineId;
      }
    };

    // Helper to show deletion feedback popup
    const showDeletionFeedback = (sentenceText, sentenceId) => {
      // Remove any existing feedback popup
      const existingPopup = overlay.querySelector('.robomate-deletion-feedback');
      if (existingPopup) existingPopup.remove();

      const feedbackPopup = document.createElement('div');
      feedbackPopup.className = 'robomate-deletion-feedback';
      feedbackPopup.innerHTML = `
        <div class="robomate-deletion-feedback-content">
          <div class="robomate-deletion-feedback-title">Why did you delete this line?</div>
          <div class="robomate-deleted-line">"${assistant.escapeHtml(sentenceText)}"</div>
          <div class="robomate-deletion-feedback-tags">
            <button class="robomate-feedback-tag" data-reason="vague">Vague / No Action</button>
            <button class="robomate-feedback-tag" data-reason="parroting">Stop Parroting</button>
            <button class="robomate-feedback-tag" data-reason="overpromised">Over-Promised</button>
            <button class="robomate-feedback-tag" data-reason="missed-repair">Missed: Repair</button>
            <button class="robomate-feedback-tag" data-reason="wordy">Too Wordy</button>
          </div>
          <button class="robomate-feedback-skip" data-reason="skip">Skip</button>
        </div>
      `;

      const modal = overlay.querySelector('.robomate-ai-modal');
      modal.appendChild(feedbackPopup);

      // Add click handlers for tags
      const tags = feedbackPopup.querySelectorAll('.robomate-feedback-tag, .robomate-feedback-skip');
      tags.forEach(tag => {
        tag.onclick = () => {
          const reason = tag.getAttribute('data-reason');

          // Store the deletion reason
          if (reason !== 'skip' && assistant.currentFeedback) {
            assistant.currentFeedback.deletionReasons.push({
              sentenceId: sentenceId,
              sentenceText: sentenceText,
              reason: reason,
              timestamp: Date.now()
            });
            console.log(`📝 Deletion reason captured: ${reason}`);
          }

          // Remove popup
          feedbackPopup.remove();
        };
      });
    };

    // Helper to delete selected line
    const deleteLine = () => {
      if (selectedLineId === null) return;

      const lineElement = overlay.querySelector(`.robomate-sentence-line[data-sentence-id="${selectedLineId}"]`);
      if (!lineElement) return;

      const sentenceText = lineElement.querySelector('.robomate-sentence-text')?.textContent || '';

      // Track the edit for feedback
      if (assistant.currentFeedback) {
        assistant.currentFeedback.edits.push({
          type: 'sentence_delete',
          sentenceId: selectedLineId,
          sentenceText: sentenceText,
          timestamp: Date.now()
        });
      }

      // Save to undo stack
      assistant.undoStack.push({
        action: 'delete',
        lineId: selectedLineId,
        lineElement: lineElement.cloneNode(true)
      });

      // Remove the line
      lineElement.remove();

      // Show deletion feedback popup
      showDeletionFeedback(sentenceText, selectedLineId);

      // Select next available line
      const remainingLines = overlay.querySelectorAll('.robomate-sentence-line');
      if (remainingLines.length > 0) {
        const nextLineId = parseInt(remainingLines[0].getAttribute('data-sentence-id'));
        selectLine(nextLineId);
      } else {
        selectedLineId = null;
      }
    };

    // Helper to undo last deletion
    const undoDelete = () => {
      if (assistant.undoStack.length === 0) return;

      const lastAction = assistant.undoStack.pop();
      if (lastAction.action === 'delete') {
        const preview = document.getElementById('robomate-reply-preview');
        const lines = preview.querySelectorAll('.robomate-sentence-line');
        const lineId = lastAction.lineId;

        // Find insertion point (restore in correct order)
        let insertBefore = null;
        for (const line of lines) {
          const id = parseInt(line.getAttribute('data-sentence-id'));
          if (id > lineId) {
            insertBefore = line;
            break;
          }
        }

        if (insertBefore) {
          preview.insertBefore(lastAction.lineElement, insertBefore);
        } else {
          preview.appendChild(lastAction.lineElement);
        }

        selectLine(lineId);
      }
    };

    // Click handlers for line numbers
    setTimeout(() => {
      const lineNums = overlay.querySelectorAll('.robomate-line-num');
      lineNums.forEach(lineNum => {
        lineNum.onclick = (e) => {
          e.stopPropagation(); // Don't trigger line delete
          const line = lineNum.closest('.robomate-sentence-line');
          const lineId = parseInt(line.getAttribute('data-sentence-id'));
          selectLine(lineId);
        };
      });

      // Click handlers for entire sentence line (delete on click)
      const sentenceLines = overlay.querySelectorAll('.robomate-sentence-line');
      sentenceLines.forEach(line => {
        line.onclick = (e) => {
          // Don't delete if clicking on line number or dropdown
          if (e.target.closest('.robomate-line-num') ||
              e.target.closest('.robomate-placeholder-select') ||
              e.target.closest('.robomate-dropdown-container')) {
            return;
          }
          const lineId = parseInt(line.getAttribute('data-sentence-id'));
          selectedLineId = lineId;
          selectLine(lineId);
          deleteLine();
        };
      });

      // Hide delete overlay when hovering over dropdowns or line numbers
      sentenceLines.forEach(line => {
        const hoverOverlay = line.querySelector('.robomate-sentence-hover-overlay');
        const dropdowns = line.querySelectorAll('.robomate-dropdown-container');
        const lineNum = line.querySelector('.robomate-line-num');

        // Hide overlay when hovering over dropdown
        dropdowns.forEach(dropdown => {
          dropdown.addEventListener('mouseenter', () => {
            hoverOverlay.style.opacity = '0';
            hoverOverlay.style.pointerEvents = 'none';
          });
          dropdown.addEventListener('mouseleave', () => {
            hoverOverlay.style.opacity = '';
            hoverOverlay.style.pointerEvents = '';
          });
        });

        // Hide overlay when hovering over line number
        if (lineNum) {
          lineNum.addEventListener('mouseenter', () => {
            hoverOverlay.style.opacity = '0';
            hoverOverlay.style.pointerEvents = 'none';
          });
          lineNum.addEventListener('mouseleave', () => {
            hoverOverlay.style.opacity = '';
            hoverOverlay.style.pointerEvents = '';
          });
        }
      });
    }, 100);

    // Keyboard navigation handler
    const handleModalKeydown = (e) => {
      // ALWAYS stop propagation to block underlying page navigation
      e.stopPropagation();
      e.stopImmediatePropagation();

      const selects = Array.from(overlay.querySelectorAll('.robomate-placeholder-select'));
      const currentIndex = selects.indexOf(document.activeElement);
      const isFocusedOnSelect = currentIndex !== -1;
      const commentTextarea = overlay.querySelector('#robomate-feedback-comment');
      const isFocusedOnComment = document.activeElement === commentTextarea;
      const manualTextarea = overlay.querySelector('#robomate-manual-textarea');
      const isFocusedOnManual = document.activeElement === manualTextarea;
      let finalText; // Declare once for reuse in multiple cases

      // If in manual edit mode or focused on comment, only handle Escape - let all other keys work normally
      if ((isFocusedOnComment || isFocusedOnManual) && e.key !== 'Escape') {
        // Already stopped propagation above to prevent background navigation
        return;
      }

      // For all other cases, prevent default behavior
      e.preventDefault();

      // Track for dd detection
      const now = Date.now();
      const isRepeatedKey = e.key === lastKeyPressed && (now - lastKeyTime) < 500;
      lastKeyPressed = e.key;
      lastKeyTime = now;

      switch(e.key) {
        case 'Escape':
          e.preventDefault();
          e.stopPropagation();
          // If focused on comment textarea, exit and return to sentence navigation
          if (isFocusedOnComment) {
            commentTextarea.blur();
            const firstSentence = overlay.querySelector('.robomate-sentence-line');
            if (firstSentence) {
              const firstLineId = parseInt(firstSentence.getAttribute('data-sentence-id'));
              selectLine(firstLineId);
            }
          } else if (isFocusedOnSelect) {
            // If focused on dropdown, exit dropdown mode and return to sentence navigation
            document.activeElement.blur();
            const firstSentence = overlay.querySelector('.robomate-sentence-line');
            if (firstSentence) {
              const firstLineId = parseInt(firstSentence.getAttribute('data-sentence-id'));
              selectLine(firstLineId);
            }
          } else {
            // Otherwise close the modal
            window.replymateModalOpen = false;
          overlay.remove();
            document.removeEventListener('keydown', handleModalKeydown, true);
            // Navigate back to ticket list if we have a return URL
            if (returnUrl) {
              sessionStorage.removeItem('replymate-vim-return-url');
              sessionStorage.setItem('replymate-vim-just-returned', 'true');
              window.location.href = returnUrl;
            }
          }
          break;

        case 'd':
          e.preventDefault();
          e.stopPropagation();
          // dd = delete line (double press)
          if (isRepeatedKey && !isFocusedOnSelect) {
            deleteLine();
          }
          break;

        case 'u':
          e.preventDefault();
          e.stopPropagation();
          // Undo
          if (!isFocusedOnSelect) {
            undoDelete();
          }
          break;

        /* COMMENTED OUT: h/l navigation keys - can resurrect later
        case 'h':
          e.preventDefault();
          e.stopPropagation();
          if (selects.length > 0) {
            let nextIndex;
            if (isFocusedOnSelect) {
              nextIndex = currentIndex - 1;
              if (nextIndex < 0) nextIndex = selects.length - 1;
            } else {
              nextIndex = selects.length - 1; // Focus last if none focused
            }
            selects[nextIndex].focus();
          }
          break;

        case 'l':
          e.preventDefault();
          e.stopPropagation();
          if (selects.length > 0) {
            let nextIndex;
            if (isFocusedOnSelect) {
              nextIndex = currentIndex + 1;
              if (nextIndex >= selects.length) nextIndex = 0;
            } else {
              nextIndex = 0; // Focus first if none focused
            }
            selects[nextIndex].focus();
          }
          break;
        END COMMENTED OUT */

        case 'c':
        case 'C':
          // Copy to clipboard
          if (!e.ctrlKey && !e.metaKey) {
            e.preventDefault();
            e.stopPropagation();
            finalText = assistant.getFinalReplyText();
            navigator.clipboard.writeText(finalText);
            const btn = document.getElementById('robomate-copy-btn');
            btn.textContent = '✓ Copied!';
            setTimeout(() => {
              window.replymateModalOpen = false;
          overlay.remove();
              document.removeEventListener('keydown', handleModalKeydown, true);
              // Navigate back to ticket list if we have a return URL
              if (returnUrl) {
                sessionStorage.removeItem('replymate-vim-return-url');
                sessionStorage.setItem('replymate-vim-just-returned', 'true');
                window.location.href = returnUrl;
              }
            }, 1000);
          }
          break;

        /* COMMENTED OUT: 'o' and 'O' handlers - can resurrect later
        case 'o':
          // Lowercase o: Insert reply (already on ticket page)
          e.preventDefault();
          e.stopPropagation();
          finalText = assistant.getFinalReplyText();
          assistant.captureFeedback('sent');
          // Store for insertion by content.js
          sessionStorage.setItem('replymate-pending-reply', finalText);
          sessionStorage.setItem('replymate-pending-ticket-id', context.ticketId);
          // Close modal - content.js will handle insertion
          window.replymateModalOpen = false;
          overlay.remove();
          document.removeEventListener('keydown', handleModalKeydown, true);
          // Trigger insertion immediately
          window.location.reload();
          break;

        case 'O':
          // Shift+O: Close modal and discard suggestion (already on ticket page)
          e.preventDefault();
          e.stopPropagation();
          assistant.captureFeedback('discarded');
          // Clear return URL since we're staying on this page
          sessionStorage.removeItem('replymate-vim-return-url');
          window.replymateModalOpen = false;
          overlay.remove();
          document.removeEventListener('keydown', handleModalKeydown, true);
          break;
        END COMMENTED OUT */

        case 'Enter':
          e.preventDefault();
          e.stopPropagation();
          finalText = assistant.getFinalReplyText();
          sessionStorage.setItem('replymate-pending-reply', finalText);
          sessionStorage.setItem('replymate-pending-ticket-id', context.ticketId);
          if (ticketUrl) {
            window.location.href = ticketUrl;
          }
          window.replymateModalOpen = false;
          overlay.remove();
          document.removeEventListener('keydown', handleModalKeydown, true);
          break;

        default:
          // All keys already have stopPropagation at top of handler
          break;
      }
    };

    document.addEventListener('keydown', handleModalKeydown, true); // Use capture phase to run before vim navigation

    // Mouse event handlers
    document.getElementById('robomate-close-btn').onclick = () => {
      assistant.captureFeedback('discarded');
      window.replymateModalOpen = false;
          overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
      // Navigate back to ticket list if we have a return URL
      if (returnUrl) {
        sessionStorage.removeItem('replymate-vim-return-url');
        sessionStorage.setItem('replymate-vim-just-returned', 'true');
        window.location.href = returnUrl;
      }
    };

    overlay.onclick = (e) => {
      if (e.target === overlay) {
        assistant.captureFeedback('discarded');
        window.replymateModalOpen = false;
          overlay.remove();
        document.removeEventListener('keydown', handleModalKeydown, true);
        // Navigate back to ticket list if we have a return URL
        if (returnUrl) {
          sessionStorage.removeItem('replymate-vim-return-url');
          sessionStorage.setItem('replymate-vim-just-returned', 'true');
          window.location.href = returnUrl;
        }
      }
    };

    document.getElementById('robomate-manual-edit-btn').onclick = () => {
      // Convert to manual edit mode
      const preview = document.getElementById('robomate-reply-preview');
      const finalText = assistant.getFinalReplyText();

      // Replace with textarea
      preview.innerHTML = `<textarea id="robomate-manual-textarea" class="robomate-manual-textarea">${finalText}</textarea>`;

      // Hide the manual edit button and help text
      document.getElementById('robomate-manual-edit-btn').style.display = 'none';
      document.querySelector('.robomate-reply-help-text').style.display = 'none';

      // Disable keyboard shortcuts (set flag)
      assistant.manualEditMode = true;

      // Update getFinalReplyText to use textarea
      assistant.getFinalReplyText = () => {
        const textarea = document.getElementById('robomate-manual-textarea');
        return textarea ? textarea.value : '';
      };
    };

    document.getElementById('robomate-copy-btn').onclick = () => {
      const finalText = assistant.getFinalReplyText();
      assistant.captureFeedback('copied');
      navigator.clipboard.writeText(finalText);
      const btn = document.getElementById('robomate-copy-btn');
      btn.textContent = '✓ Copied!';
      setTimeout(() => {
        window.replymateModalOpen = false;
          overlay.remove();
        document.removeEventListener('keydown', handleModalKeydown, true);
        // Navigate back to ticket list if we have a return URL
        if (returnUrl) {
          sessionStorage.removeItem('repomate-vim-return-url');
          sessionStorage.setItem('replymate-vim-just-returned', 'true');
          window.location.href = returnUrl;
        }
      }, 1000);
    };

    document.getElementById('robomate-insert-btn').onclick = () => {
      const finalText = assistant.getFinalReplyText();
      assistant.captureFeedback('sent');
      // Store for insertion by content.js
      sessionStorage.setItem('replymate-pending-reply', finalText);
      sessionStorage.setItem('replymate-pending-ticket-id', context.ticketId);
      window.replymateModalOpen = false;
          overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
      // Trigger insertion immediately (we're already on ticket page)
      window.location.reload();
    };

    document.getElementById('robomate-discard-btn').onclick = () => {
      assistant.captureFeedback('discarded');
      // Clear return URL since we're staying on this page
      sessionStorage.removeItem('replymate-vim-return-url');
      window.replymateModalOpen = false;
          overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
    };

    // Track dropdown changes
    const dropdowns = overlay.querySelectorAll('.robomate-placeholder-select');
    dropdowns.forEach(dropdown => {
      dropdown.addEventListener('change', (e) => {
        if (assistant.currentFeedback) {
          const placeholder = e.target.getAttribute('data-placeholder');
          const fromValue = e.target.getAttribute('data-original-value') || e.target.options[0].value;
          const toValue = e.target.value;

          // Store original value on first change
          if (!e.target.hasAttribute('data-original-value')) {
            e.target.setAttribute('data-original-value', fromValue);
          }

          assistant.currentFeedback.edits.push({
            type: 'dropdown_change',
            placeholder: placeholder,
            from: fromValue,
            to: toValue,
            timestamp: Date.now()
          });
        }
      });
    });

    // Store feedback in memory for now
    assistant.currentFeedback = {
      edits: [], // Will be populated as user edits/deletes lines
      deletionReasons: [] // Will store why lines were deleted
    };
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  checkPendingModal() {
    // Check if we have a pending modal to show (from vim navigation 'i' key)
    const pendingReply = sessionStorage.getItem('replymate-vim-modal-reply');
    const pendingContext = sessionStorage.getItem('replymate-vim-modal-context');
    const returnUrl = sessionStorage.getItem('replymate-vim-return-url');

    if (pendingReply && pendingContext) {
      // Clear the stored data
      sessionStorage.removeItem('replymate-vim-modal-reply');
      sessionStorage.removeItem('replymate-vim-modal-context');
      // Keep returnUrl for Esc handler

      // Parse context
      const context = JSON.parse(pendingContext);

      // Show the modal with the stored reply
      // We pass null for ticketUrl since we're already on the ticket page
      this.showAIResult(pendingReply, null, context, returnUrl);
    }
  }
}

// Initialize vim navigation
const vimNav = new VimNavigation();

// Check for pending modal on page load
window.addEventListener('load', () => {
  vimNav.checkPendingModal();
});
