// Replymate - Content Script
// Runs on all Freshdesk ticket pages

// Backend configuration
const BACKEND_URL = 'https://robomate-backend.thelockmeister.workers.dev';

console.log('🤖 Replymate loaded');

class FreshdeskAIAssistant {
  constructor() {
    this.config = null;
    this.buttonInjected = false;
    this.retryCount = 0;
    this.maxRetries = 10; // Try for up to 5 seconds (10 * 500ms)
    this.init();
  }

  async init() {
    // Load configuration
    await this.loadConfig();

    // Wait for page to be ready
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', () => {
        this.injectButton();
        this.checkPendingReply();
      });
    } else {
      this.injectButton();
      this.checkPendingReply();
    }

    // Watch for navigation changes (Freshdesk is a SPA)
    this.watchForChanges();
  }

  checkPendingReply() {
    // Check if there's a pending reply from vim navigation
    const pendingReply = sessionStorage.getItem('replymate-pending-reply');
    const pendingTicketId = sessionStorage.getItem('replymate-pending-ticket-id');

    if (pendingReply) {
      // Get current page ticket ID from URL
      const urlMatch = window.location.href.match(/\/tickets\/(\d+)/);
      const currentTicketId = urlMatch ? urlMatch[1] : null;

      // Verify we're on the correct ticket page
      if (pendingTicketId && currentTicketId && pendingTicketId !== currentTicketId) {
        console.warn(`⚠️ Ticket ID mismatch: pending=${pendingTicketId}, current=${currentTicketId}. Skipping insertion.`);
        sessionStorage.removeItem('replymate-pending-reply');
        sessionStorage.removeItem('replymate-pending-ticket-id');
        return;
      }

      // Clear storage
      sessionStorage.removeItem('replymate-pending-reply');
      sessionStorage.removeItem('replymate-pending-ticket-id');

      console.log(`✓ Inserting reply for ticket #${currentTicketId}`);

      // Wait for page to fully load, then open reply and insert
      setTimeout(() => {
        // Find the reply button and click it
        const replyButtons = document.querySelectorAll('button');
        for (const btn of replyButtons) {
          if (btn.textContent.includes('Reply')) {
            btn.click();

            // Wait for editor to appear, then inject
            setTimeout(() => {
              const editor = this.findReplyEditor();
              if (editor) {
                this.injectTextIntoEditor(editor, pendingReply);
                console.log('✓ Auto-inserted reply from vim navigation');
              }
            }, 500);
            break;
          }
        }
      }, 1000);
    }
  }

  async loadConfig() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['userEmail'], (result) => {
        this.config = {
          userEmail: result.userEmail || null
        };
        resolve();
      });
    });
  }

  watchForChanges() {
    // Watch for URL changes in SPA
    let lastUrl = location.href;
    new MutationObserver(() => {
      const url = location.href;
      if (url !== lastUrl) {
        lastUrl = url;
        this.buttonInjected = false;
        this.retryCount = 0; // Reset retry counter for new page
        setTimeout(() => this.injectButton(), 1000);
      }
    }).observe(document.body, { subtree: true, childList: true });

    // Also watch for reply editor appearing
    const observer = new MutationObserver(() => {
      const editor = this.findReplyEditor();
      if (editor && !this.buttonInjected) {
        this.injectButton();
      }
    });
    observer.observe(document.body, { subtree: true, childList: true });
  }

  injectButton() {
    // Only inject on ticket pages
    if (!window.location.href.includes('/tickets/')) {
      return;
    }

    // Check if button already exists
    if (document.getElementById('robomate-ai-button')) {
      this.buttonInjected = true;
      this.retryCount = 0; // Reset retry counter
      return;
    }

    // Find the toolbar - try multiple selectors
    const toolbarSelectors = [
      'button[aria-label*="Reply"]',
      'button[title*="Reply"]',
      '.page-actions',
      '.ticket-actions',
      'header button',
      '[class*="action"]'
    ];

    let toolbar = null;
    for (const selector of toolbarSelectors) {
      const elements = document.querySelectorAll(selector);
      if (elements.length > 0) {
        // Find the parent that contains the reply button
        toolbar = elements[0].closest('header, [class*="actions"], [class*="toolbar"]');
        if (!toolbar) {
          toolbar = elements[0].parentElement;
        }
        break;
      }
    }

    if (!toolbar) {
      this.retryCount++;
      if (this.retryCount <= this.maxRetries) {
        console.log(`Could not find toolbar, will retry (${this.retryCount}/${this.maxRetries})...`);
        setTimeout(() => this.injectButton(), 500);
      } else {
        console.warn('Failed to find toolbar after maximum retries');
      }
      return;
    }

    // Create AI button
    const button = document.createElement('button');
    button.id = 'robomate-ai-button';

    // Add logo image
    const logoUrl = chrome.runtime.getURL('icons/robomate-logo.png');
    button.innerHTML = `<img src="${logoUrl}" alt="Replymate" class="robomate-btn-logo">`;
    button.className = 'robomate-ai-btn';
    button.title = 'AI-powered reply suggestions by Replymate';

    button.onclick = () => this.handleButtonClick();

    // Insert button
    toolbar.appendChild(button);
    this.buttonInjected = true;
    this.retryCount = 0; // Reset retry counter on success
    console.log('✓ AI button injected');
  }

  async handleButtonClick() {
    console.log('AI button clicked');

    // Check if API key is configured
    await this.loadConfig();

    if (!this.config.apiKey) {
      this.showMessage('Please configure your OpenAI API key first. Click the extension icon in the toolbar.', 'warning');
      return;
    }

    // Find reply editor
    const editor = this.findReplyEditor();
    if (!editor) {
      this.showMessage('Please click the "Reply" button first to open the reply editor.', 'error');
      return;
    }

    // Extract ticket context
    const context = await this.extractContext();
    if (!context.customerMessage || context.customerMessage.length < 20) {
      this.showMessage('Could not extract customer message from ticket.', 'error');
      console.error('Context extraction failed:', context);
      return;
    }

    console.log('Context extracted:', context);

    // Show loading
    this.showLoading();

    try {
      // Generate reply
      const reply = await this.generateReply(context);

      // Hide loading
      this.hideLoading();

      // Show result
      this.showResult(reply, editor, context);

    } catch (error) {
      this.hideLoading();
      this.showMessage('Error: ' + error.message, 'error');
      console.error('Error generating reply:', error);
    }
  }

  findReplyEditor() {
    const selectors = [
      '[contenteditable="true"]',
      'div[contenteditable]',
      'textarea[name*="reply"]',
      '.reply-editor textarea',
      '.editor-content[contenteditable]'
    ];

    for (const selector of selectors) {
      const el = document.querySelector(selector);
      if (el && el.offsetParent !== null) { // Check if visible
        return el;
      }
    }

    return null;
  }

  async extractContext() {
    const context = {
      subject: '',
      customerMessage: '',
      customerName: '',
      customerEmail: '',
      ticketId: ''
    };

    // Extract subject
    const h1 = document.querySelector('h1');
    if (h1) {
      context.subject = h1.textContent.trim();
    }

    // Extract ticket ID from URL
    const urlMatch = window.location.href.match(/tickets\/(\d+)/);
    if (urlMatch) {
      context.ticketId = urlMatch[1];
    }

    // Extract customer name
    const nameLink = document.querySelector('a[href*="/contacts/"]');
    if (nameLink) {
      context.customerName = nameLink.textContent.trim();
    }

    // Extract customer email
    const emailPattern = /[\w.-]+@[\w.-]+\.\w+/;
    const bodyText = document.body.textContent;
    const emailMatch = bodyText.match(emailPattern);
    if (emailMatch) {
      context.customerEmail = emailMatch[0];
    }

    // Fetch latest customer message from conversations API
    if (context.ticketId) {
      try {
        const response = await fetch(`https://robomateco-help.freshdesk.com/api/v2/tickets/${context.ticketId}/conversations`, {
          headers: {
            'Authorization': 'Basic ' + btoa('szMA73aUly74WwQBUrcV:X'),
            'Content-Type': 'application/json'
          }
        });

        if (response.ok) {
          const conversations = await response.json();

          // Filter to only customer messages (incoming: true) and sort by newest first
          const customerMessages = conversations
            .filter(conv => conv.incoming === true)
            .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));

          if (customerMessages.length > 0) {
            // Get the most recent customer message
            const latestMessage = customerMessages[0];
            context.customerMessage = (latestMessage.body_text || latestMessage.body || '').trim();
            console.log(`📧 Found latest customer message from ${latestMessage.created_at}`);
          }
        }
      } catch (error) {
        console.error('Failed to fetch conversations:', error);
      }
    }

    // Fallback to DOM scraping if API call failed
    if (!context.customerMessage) {
      console.log('📝 Falling back to DOM scraping for customer message');
      const messageSelectors = [
        'blockquote',
        '[class*="message-body"]',
        '[class*="email-body"]',
        '.ticket-thread',
        '[class*="conversation"]'
      ];

      for (const selector of messageSelectors) {
        const elements = document.querySelectorAll(selector);
        for (const el of elements) {
          const text = el.textContent.trim();
          if (text.length > 50 && !text.includes('wrote:')) {
            context.customerMessage = text;
            break;
          }
        }
        if (context.customerMessage) break;
      }

      // Fallback: find quoted text
      if (!context.customerMessage) {
        const quotes = document.querySelectorAll('[style*="border-left"]');
        for (const quote of quotes) {
          const text = quote.textContent.trim();
          if (text.length > 50) {
            context.customerMessage = text;
            break;
          }
        }
      }
    }

    return context;
  }

  async generateReply(context) {
    // Check if user email is configured
    if (!this.config.userEmail) {
      throw new Error('Please configure your email in extension settings');
    }

    // Get ticket URL and ID
    const ticketUrl = window.location.href;
    const ticketId = context.ticketId || ticketUrl.match(/\/tickets\/(\d+)/)?.[1] || 'unknown';
    const freshdeskDomain = window.location.hostname;

    // Call backend API
    const response = await fetch(`${BACKEND_URL}/api/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-User-Email': this.config.userEmail
      },
      body: JSON.stringify({
        ticketId: ticketId,
        ticketUrl: ticketUrl,
        freshdeskDomain: freshdeskDomain,
        customerName: context.customerName,
        customerEmail: context.customerEmail || null,
        subject: context.subject || 'No subject',
        customerMessage: context.customerMessage,
        ticketStatus: context.status || 'Unknown'
      })
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error || `Backend request failed: ${response.status}`);
    }

    const data = await response.json();

    // Store generation ID for feedback
    this.currentGenerationId = data.generationId;
    this.currentTicketId = ticketId;

    return data.reply;
  }

  showLoading() {
    const existing = document.getElementById('robomate-ai-loading');
    if (existing) existing.remove();

    const overlay = document.createElement('div');
    overlay.id = 'robomate-ai-loading';
    overlay.className = 'robomate-ai-overlay';
    overlay.innerHTML = `
      <div class="robomate-ai-modal">
        <div class="robomate-ai-header">
          <div class="robomate-ai-title">Replymate</div>
        </div>
        <div class="robomate-ai-body">
          <div class="robomate-ai-loading-text">Analyzing ticket and generating reply...</div>
          <div class="robomate-ai-progress">
            <div class="robomate-ai-progress-bar"></div>
          </div>
        </div>
      </div>
    `;
    document.body.appendChild(overlay);
  }

  hideLoading() {
    const loading = document.getElementById('robomate-ai-loading');
    if (loading) loading.remove();
  }

  showResult(reply, editor, context) {
    const existing = document.getElementById('robomate-ai-result');
    if (existing) existing.remove();

    // Convert placeholders to interactive dropdowns
    const interactiveHtml = this.createInteractiveReply(reply);

    const overlay = document.createElement('div');
    overlay.id = 'robomate-ai-result';
    overlay.className = 'robomate-ai-overlay';
    overlay.innerHTML = `
      <div class="robomate-ai-modal robomate-ai-modal-large">
        <div class="robomate-ai-header">
          <div class="robomate-ai-title">Replymate Suggestion</div>
          <button class="robomate-ai-close" id="robomate-close-btn">×</button>
        </div>
        <div class="robomate-ai-body">
          <div class="robomate-top-row">
            <div class="robomate-customer-message">
              <div class="robomate-message-header">
                <span class="robomate-message-icon">💬</span>
                <span class="robomate-message-from">Message from ${this.escapeHtml(context.customerName || 'Customer')}</span>
              </div>
              <div class="robomate-message-content">${this.escapeHtml(context.customerMessage || 'No message content')}</div>
            </div>
            <!-- COMMENTED OUT: Keyboard navigation compass - can resurrect later
            <div class="robomate-vim-compass">
              <div class="robomate-compass-title">Keyboard Navigation</div>
              <div class="robomate-compass-rose">
                <div class="robomate-compass-key robomate-compass-up">
                  <span class="robomate-key">k</span>
                  <span class="robomate-label">UP</span>
                </div>
                <div class="robomate-compass-center">
                  <div class="robomate-compass-key robomate-compass-left">
                    <span class="robomate-key">h</span>
                    <span class="robomate-label">PREV DROPDOWN</span>
                  </div>
                  <div class="robomate-compass-middle">
                    <span class="robomate-vim-text">VIM</span>
                  </div>
                  <div class="robomate-compass-key robomate-compass-right">
                    <span class="robomate-key">l</span>
                    <span class="robomate-label">NEXT DROPDOWN</span>
                  </div>
                </div>
                <div class="robomate-compass-key robomate-compass-down">
                  <span class="robomate-key">j</span>
                  <span class="robomate-label">DOWN</span>
                </div>
              </div>
              <div class="robomate-compass-actions">
                <span><strong>dd</strong> delete line</span> •
                <span><strong>u</strong> undo</span> •
                <span><strong>c</strong> copy</span> •
                <span><strong>i</strong> insert</span> •
                <span><strong>Esc</strong> close</span>
              </div>
            </div>
            END COMMENTED OUT -->
          </div>
          <div class="robomate-our-reply-section">
            <div class="robomate-reply-header">
              <span class="robomate-reply-title">Our Reply</span>
              <button class="robomate-manual-edit-btn" id="robomate-manual-edit-btn" title="Switch to manual editing mode">Manual Edit</button>
              <span class="robomate-reply-help-text">ℹ️ Hover lines to delete</span>
            </div>
            <div class="robomate-ai-reply-preview" id="robomate-reply-preview">${interactiveHtml}</div>
          </div>
          <div class="robomate-feedback-comment-section">
            <textarea
              class="robomate-feedback-comment"
              id="robomate-feedback-comment"
              placeholder="Optional: Add any comments about this generation..."
              rows="2"></textarea>
          </div>
          <div class="robomate-ai-actions">
            <button class="robomate-ai-btn-secondary" id="robomate-copy-btn">📋 Copy to Clipboard</button>
            <button class="robomate-ai-btn-tertiary" id="robomate-discard-btn">✕ Open Ticket (Discard Suggestion)</button>
            <button class="robomate-ai-btn-primary" id="robomate-insert-btn">✓ Insert into Reply</button>
          </div>
        </div>
      </div>
    `;

    document.body.appendChild(overlay);
    
    // Set global flag to block vim navigation
    window.replymateModalOpen = true;

    // Capture this context
    const self = this;

    // State for sentence line editing
    let selectedLineId = null;
    let lastKeyPressed = null;
    let lastKeyTime = 0;

    // Helper to select a sentence line
    const selectLine = (lineId) => {
      const lines = overlay.querySelectorAll('.robomate-sentence-line');
      lines.forEach(line => line.classList.remove('selected'));
      const targetLine = overlay.querySelector(`.robomate-sentence-line[data-sentence-id="${lineId}"]`);
      if (targetLine) {
        targetLine.classList.add('selected');
        targetLine.scrollIntoView({ block: 'nearest', behavior: 'smooth' });
        selectedLineId = lineId;
      }
    };

    // Helper to show deletion feedback popup
    const showDeletionFeedback = (sentenceText, sentenceId) => {
      // Remove any existing feedback popup
      const existingPopup = overlay.querySelector('.robomate-deletion-feedback');
      if (existingPopup) existingPopup.remove();

      const feedbackPopup = document.createElement('div');
      feedbackPopup.className = 'robomate-deletion-feedback';
      feedbackPopup.innerHTML = `
        <div class="robomate-deletion-feedback-content">
          <div class="robomate-deletion-feedback-title">Why did you delete this line?</div>
          <div class="robomate-deleted-line">"${self.escapeHtml(sentenceText)}"</div>
          <div class="robomate-deletion-feedback-tags">
            <button class="robomate-feedback-tag" data-reason="vague">Vague / No Action</button>
            <button class="robomate-feedback-tag" data-reason="parroting">Stop Parroting</button>
            <button class="robomate-feedback-tag" data-reason="overpromised">Over-Promised</button>
            <button class="robomate-feedback-tag" data-reason="missed-repair">Missed: Repair</button>
            <button class="robomate-feedback-tag" data-reason="wordy">Too Wordy</button>
          </div>
          <button class="robomate-feedback-skip" data-reason="skip">Skip</button>
        </div>
      `;

      const modal = overlay.querySelector('.robomate-ai-modal');
      modal.appendChild(feedbackPopup);

      // Add click handlers for tags
      const tags = feedbackPopup.querySelectorAll('.robomate-feedback-tag, .robomate-feedback-skip');
      tags.forEach(tag => {
        tag.onclick = () => {
          const reason = tag.getAttribute('data-reason');

          // Store the deletion reason
          if (reason !== 'skip' && self.currentFeedback) {
            self.currentFeedback.deletionReasons.push({
              sentenceId: sentenceId,
              sentenceText: sentenceText,
              reason: reason,
              timestamp: Date.now()
            });
            console.log(`📝 Deletion reason captured: ${reason}`);
          }

          // Remove popup
          feedbackPopup.remove();
        };
      });
    };

    // Helper to delete selected line
    const deleteLine = () => {
      if (selectedLineId === null) return;

      const lineElement = overlay.querySelector(`.robomate-sentence-line[data-sentence-id="${selectedLineId}"]`);
      if (!lineElement) return;

      const sentenceText = lineElement.querySelector('.robomate-sentence-text')?.textContent || '';

      // Track the edit for feedback
      if (self.currentFeedback) {
        self.currentFeedback.edits.push({
          type: 'sentence_delete',
          sentenceId: selectedLineId,
          sentenceText: sentenceText,
          timestamp: Date.now()
        });
      }

      // Save to undo stack
      self.undoStack.push({
        action: 'delete',
        lineId: selectedLineId,
        lineElement: lineElement.cloneNode(true)
      });

      // Remove the line
      lineElement.remove();

      // Show deletion feedback popup
      showDeletionFeedback(sentenceText, selectedLineId);

      // Select next available line
      const remainingLines = overlay.querySelectorAll('.robomate-sentence-line');
      if (remainingLines.length > 0) {
        const nextLineId = parseInt(remainingLines[0].getAttribute('data-sentence-id'));
        selectLine(nextLineId);
      } else {
        selectedLineId = null;
      }
    };

    // Helper to undo last deletion
    const undoDelete = () => {
      if (self.undoStack.length === 0) return;

      const lastAction = self.undoStack.pop();
      if (lastAction.action === 'delete') {
        const preview = document.getElementById('robomate-reply-preview');
        const lines = preview.querySelectorAll('.robomate-sentence-line');
        const lineId = lastAction.lineId;

        // Find insertion point (restore in correct order)
        let insertBefore = null;
        for (const line of lines) {
          const id = parseInt(line.getAttribute('data-sentence-id'));
          if (id > lineId) {
            insertBefore = line;
            break;
          }
        }

        if (insertBefore) {
          preview.insertBefore(lastAction.lineElement, insertBefore);
        } else {
          preview.appendChild(lastAction.lineElement);
        }

        selectLine(lineId);
      }
    };

    // Click handlers for line numbers and delete icons
    setTimeout(() => {
      const lineNums = overlay.querySelectorAll('.robomate-line-num');
      lineNums.forEach(lineNum => {
        lineNum.onclick = (e) => {
          e.stopPropagation(); // Don't trigger line delete
          const line = lineNum.closest('.robomate-sentence-line');
          const lineId = parseInt(line.getAttribute('data-sentence-id'));
          selectLine(lineId);
        };
      });

      // Click handlers for entire sentence line (delete on click)
      const sentenceLines = overlay.querySelectorAll('.robomate-sentence-line');
      sentenceLines.forEach(line => {
        line.onclick = (e) => {
          // Don't delete if clicking on line number or dropdown
          if (e.target.closest('.robomate-line-num') ||
              e.target.closest('.robomate-placeholder-select') ||
              e.target.closest('.robomate-dropdown-container')) {
            return;
          }
          const lineId = parseInt(line.getAttribute('data-sentence-id'));
          selectedLineId = lineId;
          selectLine(lineId);
          deleteLine();
        };
      });

      // Hide delete overlay when hovering over dropdowns or line numbers
      sentenceLines.forEach(line => {
        const hoverOverlay = line.querySelector('.robomate-sentence-hover-overlay');
        const dropdowns = line.querySelectorAll('.robomate-dropdown-container');
        const lineNum = line.querySelector('.robomate-line-num');

        // Hide overlay when hovering over dropdown
        dropdowns.forEach(dropdown => {
          dropdown.addEventListener('mouseenter', () => {
            hoverOverlay.style.opacity = '0';
            hoverOverlay.style.pointerEvents = 'none';
          });
          dropdown.addEventListener('mouseleave', () => {
            hoverOverlay.style.opacity = '';
            hoverOverlay.style.pointerEvents = '';
          });
        });

        // Hide overlay when hovering over line number
        if (lineNum) {
          lineNum.addEventListener('mouseenter', () => {
            hoverOverlay.style.opacity = '0';
            hoverOverlay.style.pointerEvents = 'none';
          });
          lineNum.addEventListener('mouseleave', () => {
            hoverOverlay.style.opacity = '';
            hoverOverlay.style.pointerEvents = '';
          });
        }
      });
    }, 100);

    // Keyboard navigation handler
    const handleModalKeydown = (e) => {
      // ALWAYS stop propagation to block underlying page navigation
      e.stopPropagation();
      e.stopImmediatePropagation();

      const selects = Array.from(overlay.querySelectorAll('.robomate-placeholder-select'));
      const currentIndex = selects.indexOf(document.activeElement);
      const isFocusedOnSelect = currentIndex !== -1;
      const commentTextarea = overlay.querySelector('#robomate-feedback-comment');
      const isFocusedOnComment = document.activeElement === commentTextarea;
      const manualTextarea = overlay.querySelector('#robomate-manual-textarea');
      const isFocusedOnManual = document.activeElement === manualTextarea;

      // If in manual edit mode or focused on comment, only handle Escape - let all other keys work normally
      if ((isFocusedOnComment || isFocusedOnManual) && e.key !== 'Escape') {
        // Already stopped propagation above to prevent background navigation
        return;
      }

      // For all other cases, prevent default behavior
      e.preventDefault();

      // Track for dd detection
      const now = Date.now();
      const isRepeatedKey = e.key === lastKeyPressed && (now - lastKeyTime) < 500;
      lastKeyPressed = e.key;
      lastKeyTime = now;

      switch(e.key) {
        case 'Escape':
          e.preventDefault();
          e.stopPropagation();
          // If focused on comment textarea, exit and return to sentence navigation
          if (isFocusedOnComment) {
            commentTextarea.blur();
            const firstSentence = overlay.querySelector('.robomate-sentence-line');
            if (firstSentence) {
              const firstLineId = parseInt(firstSentence.getAttribute('data-sentence-id'));
              selectLine(firstLineId);
            }
          } else if (isFocusedOnSelect) {
            // If focused on dropdown, exit dropdown mode and return to sentence navigation
            document.activeElement.blur();
            const firstSentence = overlay.querySelector('.robomate-sentence-line');
            if (firstSentence) {
              const firstLineId = parseInt(firstSentence.getAttribute('data-sentence-id'));
              selectLine(firstLineId);
            }
          } else {
            // Otherwise close the modal
            window.replymateModalOpen = false;
            overlay.remove();
            document.removeEventListener('keydown', handleModalKeydown, true);
          }
          break;

        case 'd':
          e.preventDefault();
          e.stopPropagation();
          // dd = delete line (double press)
          if (isRepeatedKey && !isFocusedOnSelect) {
            deleteLine();
          }
          break;

        case 'u':
          e.preventDefault();
          e.stopPropagation();
          // Undo
          if (!isFocusedOnSelect) {
            undoDelete();
          }
          break;

        /* COMMENTED OUT: h/l navigation keys - can resurrect later
        case 'h':
          e.preventDefault();
          e.stopPropagation();
          if (selects.length > 0) {
            let nextIndex;
            if (isFocusedOnSelect) {
              nextIndex = currentIndex - 1;
              if (nextIndex < 0) nextIndex = selects.length - 1;
            } else {
              nextIndex = selects.length - 1; // Focus last if none focused
            }
            selects[nextIndex].focus();
          }
          break;

        case 'l':
          e.preventDefault();
          e.stopPropagation();
          if (selects.length > 0) {
            let nextIndex;
            if (isFocusedOnSelect) {
              nextIndex = currentIndex + 1;
              if (nextIndex >= selects.length) nextIndex = 0;
            } else {
              nextIndex = 0; // Focus first if none focused
            }
            selects[nextIndex].focus();
          }
          break;
        END COMMENTED OUT */

        case 'c':
        case 'C':
          // Copy to clipboard
          if (!e.ctrlKey && !e.metaKey) {
            e.preventDefault();
            e.stopPropagation();
            const finalText = self.getFinalReplyText();
            navigator.clipboard.writeText(finalText);
            const btn = document.getElementById('robomate-copy-btn');
            btn.textContent = '✓ Copied!';
            setTimeout(() => {
              window.replymateModalOpen = false;
              overlay.remove();
              document.removeEventListener('keydown', handleModalKeydown, true);
            }, 1000);
          }
          break;

        case 'i':
        case 'I':
        case 'Enter':
          // Insert into editor
          e.preventDefault();
          e.stopPropagation();
          const finalText = self.getFinalReplyText();
          self.captureFeedback('sent');
          self.injectTextIntoEditor(editor, finalText);
          window.replymateModalOpen = false;
          overlay.remove();
          document.removeEventListener('keydown', handleModalKeydown, true);
          break;

        /* COMMENTED OUT: Shift+O handler - can resurrect later
        case 'O':
          // Shift+O: Open ticket but discard suggestion
          e.preventDefault();
          e.stopPropagation();
          self.captureFeedback('discarded');
          window.replymateModalOpen = false;
          overlay.remove();
          document.removeEventListener('keydown', handleModalKeydown, true);
          break;
        END COMMENTED OUT */

        default:
          // All keys already have stopPropagation at top of handler
          break;
      }
    };

    document.addEventListener('keydown', handleModalKeydown, true); // Use capture phase

    // Mouse event handlers
    document.getElementById('robomate-close-btn').onclick = () => {
      self.captureFeedback('discarded');
      window.replymateModalOpen = false;
      overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
    };

    overlay.onclick = (e) => {
      if (e.target === overlay) {
        self.captureFeedback('discarded');
        window.replymateModalOpen = false;
        overlay.remove();
        document.removeEventListener('keydown', handleModalKeydown, true);
      }
    };

    // Track dropdown changes
    const dropdowns = overlay.querySelectorAll('.robomate-placeholder-select');
    dropdowns.forEach(dropdown => {
      dropdown.addEventListener('change', (e) => {
        if (self.currentFeedback) {
          const placeholder = e.target.getAttribute('data-placeholder');
          const fromValue = e.target.getAttribute('data-original-value') || e.target.options[0].value;
          const toValue = e.target.value;

          // Store original value on first change
          if (!e.target.hasAttribute('data-original-value')) {
            e.target.setAttribute('data-original-value', fromValue);
          }

          self.currentFeedback.edits.push({
            type: 'dropdown_change',
            placeholder: placeholder,
            from: fromValue,
            to: toValue,
            timestamp: Date.now()
          });
        }
      });
    });

    document.getElementById('robomate-manual-edit-btn').onclick = () => {
      // Convert to manual edit mode
      const preview = document.getElementById('robomate-reply-preview');
      const finalText = self.getFinalReplyText();

      // Replace with textarea
      preview.innerHTML = `<textarea id="robomate-manual-textarea" class="robomate-manual-textarea">${finalText}</textarea>`;

      // Hide the manual edit button and help text
      document.getElementById('robomate-manual-edit-btn').style.display = 'none';
      document.querySelector('.robomate-reply-help-text').style.display = 'none';

      // Disable keyboard shortcuts (set flag)
      self.manualEditMode = true;

      // Update getFinalReplyText to use textarea
      self.getFinalReplyText = () => {
        const textarea = document.getElementById('robomate-manual-textarea');
        return textarea ? textarea.value : '';
      };
    };

    document.getElementById('robomate-copy-btn').onclick = () => {
      const finalText = self.getFinalReplyText();
      self.captureFeedback('copied');
      navigator.clipboard.writeText(finalText);
      const btn = document.getElementById('robomate-copy-btn');
      btn.textContent = '✓ Copied!';
      setTimeout(() => {
        window.replymateModalOpen = false;
        overlay.remove();
        document.removeEventListener('keydown', handleModalKeydown, true);
      }, 1000);
    };

    document.getElementById('robomate-insert-btn').onclick = () => {
      const finalText = self.getFinalReplyText();
      self.captureFeedback('sent');
      self.injectTextIntoEditor(editor, finalText);
      window.replymateModalOpen = false;
      overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
    };

    document.getElementById('robomate-discard-btn').onclick = () => {
      self.captureFeedback('discarded');
      window.replymateModalOpen = false;
      overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
    };

    // Store feedback in memory for now
    self.currentFeedback = {
      edits: [], // Will be populated as user edits/deletes lines
      deletionReasons: [] // Will store why lines were deleted
    };
  }

  async captureFeedback(action) {
    if (!this.currentFeedback) return;
    if (!this.config.userEmail) return;

    const commentText = document.getElementById('robomate-feedback-comment')?.value || '';

    const feedback = {
      generationId: this.currentGenerationId || 'unknown',
      ticketId: this.currentTicketId || 'unknown',
      ticketUrl: window.location.href,
      timestamp: Date.now(),
      action: action, // 'sent', 'discarded'
      comment: commentText,
      edits: this.currentFeedback.edits || [],
      deletionReasons: this.currentFeedback.deletionReasons || [],
      discardReason: this.currentFeedback.discardReason || null,
      discardComment: this.currentFeedback.discardComment || null,
      originalReply: this.currentReply || '',
      finalReply: this.getFinalReplyText()
    };

    console.log('📊 Sending feedback to backend:', feedback);

    // Send feedback to backend
    try {
      const response = await fetch(`${BACKEND_URL}/api/feedback`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-User-Email': this.config.userEmail
        },
        body: JSON.stringify(feedback)
      });

      if (!response.ok) {
        console.error('Failed to submit feedback:', await response.text());
      } else {
        console.log('✓ Feedback submitted successfully');
      }
    } catch (error) {
      console.error('Error submitting feedback:', error);
    }

    return feedback;
  }

  parseSentences(text) {
    // Split by sentence-ending punctuation followed by space/newline
    const sentences = [];
    const parts = text.split(/([.!?]+[\s\n]+)/);

    let currentSentence = '';
    for (let i = 0; i < parts.length; i++) {
      currentSentence += parts[i];
      // If this part is punctuation+space, or we're at the end
      if (parts[i].match(/[.!?]+[\s\n]+/) || i === parts.length - 1) {
        const trimmed = currentSentence.trim();
        if (trimmed) {
          sentences.push({
            id: sentences.length,
            text: trimmed,
            deleted: false
          });
        }
        currentSentence = '';
      }
    }

    return sentences;
  }

  createInteractiveReply(reply) {
    // Store original reply for sentence parsing
    this.currentReply = reply;
    this.sentences = this.parseSentences(reply);
    this.undoStack = [];

    // Define placeholder options - keep values short to complete sentences naturally
    const placeholderOptions = {
      'AVAILABILITY': [
        '[AVAILABILITY]',
        'available',
        'not currently available',
        'available within 1 week',
        'available within 2 weeks',
        'available within 3 weeks',
        'available within 4 weeks',
        'available within 5 weeks',
        'available within 6 weeks'
      ],
      'DELIVERY TIME': [
        '[DELIVERY TIME]',
        '1-2 business days',
        '3-5 business days',
        '1 week',
        '1-2 weeks',
        '2-3 weeks',
        '3-4 weeks'
      ],
      'SHIP STATUS': [
        '[SHIP STATUS]',
        'has shipped',
        'is being processed',
        'will ship today',
        'will ship tomorrow',
        'will ship within 2-3 days',
        'will ship within 1 week'
      ]
    };

    // Build HTML with sentence-based lines
    let html = '';
    this.sentences.forEach(sentence => {
      if (sentence.deleted) return;

      let sentenceHtml = this.escapeHtml(sentence.text);

      // Replace placeholders with dropdowns
      for (const [placeholder, options] of Object.entries(placeholderOptions)) {
        const pattern = new RegExp(`\\[${placeholder}\\]`, 'g');
        if (sentenceHtml.includes(`[${placeholder}]`)) {
          const selectHtml = `<span class="robomate-dropdown-container"><select class="robomate-placeholder-select" data-placeholder="${placeholder}">
            ${options.map((opt, i) => `<option value="${this.escapeHtml(opt)}" ${i === 0 ? 'selected' : ''}>${this.escapeHtml(opt)}</option>`).join('')}
          </select></span>`;
          sentenceHtml = sentenceHtml.replace(pattern, selectHtml);
        }
      }

      html += `<div class="robomate-sentence-line" data-sentence-id="${sentence.id}">
        <div class="robomate-sentence-hover-overlay">
          <span class="robomate-hover-delete-text">🗑️ Click to delete line</span>
        </div>
        <span class="robomate-line-num">${sentence.id + 1}</span>
        <span class="robomate-sentence-text">${sentenceHtml}</span>
      </div>`;
    });

    return html;
  }

  getFinalReplyText() {
    const preview = document.getElementById('robomate-reply-preview');
    const sentenceLines = preview.querySelectorAll('.robomate-sentence-line');
    const sentences = [];

    sentenceLines.forEach(line => {
      const sentenceText = line.querySelector('.robomate-sentence-text');
      let text = sentenceText.innerHTML;

      // Replace all dropdown containers (select + help text) with their selected values
      const dropdownContainers = sentenceText.querySelectorAll('.robomate-dropdown-container');
      dropdownContainers.forEach(container => {
        const select = container.querySelector('.robomate-placeholder-select');
        if (select) {
          const selectedValue = select.options[select.selectedIndex].value;
          text = text.replace(container.outerHTML, selectedValue);
        }
      });

      // Decode HTML entities
      const tempDiv = document.createElement('div');
      tempDiv.innerHTML = text;
      sentences.push(tempDiv.textContent);
    });

    // Join sentences with space
    return sentences.join(' ');
  }

  injectTextIntoEditor(editor, text) {
    if (editor.contentEditable === 'true') {
      // ContentEditable div
      editor.innerHTML = text.replace(/\n/g, '<br>');
      editor.dispatchEvent(new Event('input', { bubbles: true }));
      editor.dispatchEvent(new Event('change', { bubbles: true }));
    } else {
      // Textarea
      editor.value = text;
      editor.dispatchEvent(new Event('input', { bubbles: true }));
      editor.dispatchEvent(new Event('change', { bubbles: true }));
    }

    // Focus the editor
    editor.focus();

    console.log('✓ Reply inserted into editor');
  }

  showMessage(message, type = 'info') {
    const existing = document.getElementById('robomate-ai-message');
    if (existing) existing.remove();

    const toast = document.createElement('div');
    toast.id = 'robomate-ai-message';
    toast.className = `robomate-ai-toast robomate-ai-toast-${type}`;
    toast.textContent = message;

    document.body.appendChild(toast);

    setTimeout(() => {
      toast.classList.add('robomate-ai-toast-show');
    }, 10);

    setTimeout(() => {
      toast.classList.remove('robomate-ai-toast-show');
      setTimeout(() => toast.remove(), 300);
    }, 5000);
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }
}

// General Feedback Widget
class GeneralFeedbackWidget {
  constructor() {
    this.config = null;
    this.isExpanded = false;
    this.init();
  }

  async init() {
    // Load configuration
    await this.loadConfig();

    // Only show on Freshdesk pages
    if (!window.location.href.includes('freshdesk.com')) {
      return;
    }

    // Inject the widget
    this.injectWidget();
  }

  async loadConfig() {
    return new Promise((resolve) => {
      chrome.storage.sync.get(['userEmail'], (result) => {
        this.config = {
          userEmail: result.userEmail || null
        };
        resolve();
      });
    });
  }

  injectWidget() {
    // Check if already exists
    if (document.getElementById('replymate-general-feedback-widget')) {
      return;
    }

    const logoUrl = chrome.runtime.getURL('icons/robomate-logo.png');

    const widget = document.createElement('div');
    widget.id = 'replymate-general-feedback-widget';
    widget.className = 'replymate-general-feedback-widget';
    widget.innerHTML = `
      <div class="replymate-feedback-toggle" id="replymate-feedback-toggle">
        <img src="${logoUrl}" class="replymate-feedback-logo" alt="Replymate">
        <span class="replymate-feedback-toggle-text">Feedback</span>
      </div>
      <div class="replymate-feedback-panel" id="replymate-feedback-panel" style="display: none;">
        <div class="replymate-feedback-panel-header">
          <img src="${logoUrl}" class="replymate-feedback-logo" alt="Replymate">
          <span>Send Feedback about Replymate</span>
          <button class="replymate-feedback-close" id="replymate-feedback-close">✕</button>
        </div>
        <textarea
          id="replymate-feedback-textarea"
          class="replymate-feedback-textarea"
          placeholder="Share your thoughts, ideas, or issues with Replymate..."
          rows="4"
        ></textarea>
        <button class="replymate-feedback-send" id="replymate-feedback-send">Send Feedback</button>
        <div class="replymate-feedback-status" id="replymate-feedback-status"></div>
      </div>
    `;

    document.body.appendChild(widget);

    // Add event listeners
    document.getElementById('replymate-feedback-toggle').onclick = () => this.togglePanel();
    document.getElementById('replymate-feedback-close').onclick = () => this.closePanel();
    document.getElementById('replymate-feedback-send').onclick = () => this.sendFeedback();

    // Close on Escape
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape' && this.isExpanded) {
        this.closePanel();
      }
    });
  }

  togglePanel() {
    const panel = document.getElementById('replymate-feedback-panel');
    const toggle = document.getElementById('replymate-feedback-toggle');

    if (this.isExpanded) {
      this.closePanel();
    } else {
      panel.style.display = 'block';
      toggle.style.display = 'none';
      this.isExpanded = true;
      document.getElementById('replymate-feedback-textarea').focus();
    }
  }

  closePanel() {
    const panel = document.getElementById('replymate-feedback-panel');
    const toggle = document.getElementById('replymate-feedback-toggle');

    panel.style.display = 'none';
    toggle.style.display = 'flex';
    this.isExpanded = false;

    // Clear status
    const status = document.getElementById('replymate-feedback-status');
    status.textContent = '';
    status.className = 'replymate-feedback-status';
  }

  async sendFeedback() {
    const textarea = document.getElementById('replymate-feedback-textarea');
    const feedbackText = textarea.value.trim();
    const status = document.getElementById('replymate-feedback-status');
    const sendButton = document.getElementById('replymate-feedback-send');

    if (!feedbackText) {
      status.textContent = 'Please enter some feedback';
      status.className = 'replymate-feedback-status error';
      return;
    }

    if (!this.config.userEmail) {
      status.textContent = 'User email not configured';
      status.className = 'replymate-feedback-status error';
      return;
    }

    // Disable button and show loading
    sendButton.disabled = true;
    sendButton.textContent = 'Sending...';
    status.textContent = '';

    try {
      const response = await fetch(`${BACKEND_URL}/api/general-feedback`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-User-Email': this.config.userEmail
        },
        body: JSON.stringify({
          feedbackText: feedbackText,
          pageUrl: window.location.href
        })
      });

      if (response.ok) {
        status.textContent = '✓ Feedback sent! Thank you!';
        status.className = 'replymate-feedback-status success';
        textarea.value = '';

        // Close after 2 seconds
        setTimeout(() => {
          this.closePanel();
        }, 2000);
      } else {
        const error = await response.json();
        status.textContent = `Error: ${error.error || 'Failed to send feedback'}`;
        status.className = 'replymate-feedback-status error';
      }
    } catch (error) {
      console.error('Feedback error:', error);
      status.textContent = 'Network error. Please try again.';
      status.className = 'replymate-feedback-status error';
    } finally {
      sendButton.disabled = false;
      sendButton.textContent = 'Send Feedback';
    }
  }
}

// Initialize the assistant
const assistant = new FreshdeskAIAssistant();
// Initialize the general feedback widget
const feedbackWidget = new GeneralFeedbackWidget();
