# Discard Feedback Feature

## Overview

When users discard an AI generation (click X or press Escape), capture WHY they discarded it.

## User Flow

```
User clicks X or Esc
        ↓
Show "Why are you discarding?" popup
        ↓
User selects reason or types free-form
        ↓
Store discard reason in feedback
        ↓
Close modal
```

## UI Design

### Discard Feedback Popup

Similar style to deletion feedback popup, but for entire generation:

```html
<div class="robomate-discard-feedback">
  <div class="robomate-discard-feedback-content">
    <div class="robomate-discard-feedback-title">
      Why are you discarding this AI suggestion?
    </div>

    <div class="robomate-discard-feedback-options">
      <button class="robomate-discard-tag" data-reason="not-relevant">
        Not Relevant to Issue
      </button>
      <button class="robomate-discard-tag" data-reason="wrong-tone">
        Wrong Tone (too formal/informal)
      </button>
      <button class="robomate-discard-tag" data-reason="factually-wrong">
        Factually Incorrect
      </button>
      <button class="robomate-discard-tag" data-reason="missing-info">
        Missing Key Information
      </button>
      <button class="robomate-discard-tag" data-reason="other">
        Other (explain below)
      </button>
    </div>

    <textarea
      class="robomate-discard-comment"
      placeholder="Optional: Tell us more..."
      rows="3"
    ></textarea>

    <div class="robomate-discard-actions">
      <button class="robomate-discard-skip">Skip</button>
      <button class="robomate-discard-submit">Submit & Close</button>
    </div>
  </div>
</div>
```

## Implementation

### 1. Update Feedback Data Structure

**Add to feedback object:**

```javascript
{
  generationId: "gen_abc123",
  action: "discarded",

  // NEW: Discard-specific feedback
  discardReason: "not-relevant",  // Tag selected
  discardComment: "Customer already received tracking info",  // Free-form text

  // Existing fields
  edits: [...],
  deletionReasons: [...],
  comment: "..."
}
```

### 2. Update Database Schema

**Add to feedback table:**

```sql
CREATE TABLE feedback (
  id TEXT PRIMARY KEY,
  generation_id TEXT NOT NULL,
  user_id TEXT NOT NULL,
  ticket_id TEXT NOT NULL,
  ticket_url TEXT NOT NULL,

  action TEXT NOT NULL,

  -- NEW: Discard feedback
  discard_reason TEXT,           -- Tag: not-relevant, wrong-tone, etc.
  discard_comment TEXT,          -- Free-form explanation

  -- Existing fields
  original_reply TEXT NOT NULL,
  final_reply TEXT NOT NULL,
  edits JSON,
  deletion_reasons JSON,
  comment TEXT,

  created_at INTEGER NOT NULL,
  FOREIGN KEY (generation_id) REFERENCES generations(id),
  FOREIGN KEY (user_id) REFERENCES users(id)
);
```

### 3. Update Close/Escape Handlers

**Before (content.js):**

```javascript
// Old: Immediate close
document.getElementById('robomate-close-btn').onclick = () => {
  self.captureFeedback('discarded');
  overlay.remove();
};
```

**After:**

```javascript
document.getElementById('robomate-close-btn').onclick = () => {
  // Show discard feedback popup BEFORE closing
  self.showDiscardFeedback(overlay, () => {
    self.captureFeedback('discarded');
    overlay.remove();
  });
};

// Same for Escape key
case 'Escape':
  if (!isFocusedOnComment && !isFocusedOnManual) {
    self.showDiscardFeedback(overlay, () => {
      self.captureFeedback('discarded');
      window.replymateModalOpen = false;
      overlay.remove();
      document.removeEventListener('keydown', handleModalKeydown, true);
    });
  }
  break;
```

### 4. Add showDiscardFeedback() Method

```javascript
showDiscardFeedback(overlay, onComplete) {
  // Create discard feedback popup
  const feedbackPopup = document.createElement('div');
  feedbackPopup.className = 'robomate-discard-feedback';
  feedbackPopup.innerHTML = `
    <div class="robomate-discard-feedback-content">
      <div class="robomate-discard-feedback-title">
        Why are you discarding this AI suggestion?
      </div>

      <div class="robomate-discard-feedback-options">
        <button class="robomate-discard-tag" data-reason="not-relevant">
          📋 Not Relevant to Issue
        </button>
        <button class="robomate-discard-tag" data-reason="wrong-tone">
          🎭 Wrong Tone
        </button>
        <button class="robomate-discard-tag" data-reason="factually-wrong">
          ❌ Factually Incorrect
        </button>
        <button class="robomate-discard-tag" data-reason="missing-info">
          ℹ️ Missing Key Info
        </button>
        <button class="robomate-discard-tag" data-reason="prefer-manual">
          ✍️ Prefer Writing Myself
        </button>
      </div>

      <textarea
        class="robomate-discard-comment"
        placeholder="Optional: Tell us more about why you're not using this suggestion..."
        rows="3"
      ></textarea>

      <div class="robomate-discard-actions">
        <button class="robomate-discard-skip">Skip & Close</button>
        <button class="robomate-discard-submit">Submit Feedback</button>
      </div>
    </div>
  `;

  overlay.appendChild(feedbackPopup);

  // Store selected reason
  let selectedReason = null;

  // Tag click handlers
  const tags = feedbackPopup.querySelectorAll('.robomate-discard-tag');
  tags.forEach(tag => {
    tag.onclick = () => {
      // Remove selected from all
      tags.forEach(t => t.classList.remove('selected'));
      // Add to clicked
      tag.classList.add('selected');
      selectedReason = tag.getAttribute('data-reason');
    };
  });

  // Skip button - close without feedback
  feedbackPopup.querySelector('.robomate-discard-skip').onclick = () => {
    feedbackPopup.remove();
    onComplete();
  };

  // Submit button - store feedback then close
  feedbackPopup.querySelector('.robomate-discard-submit').onclick = () => {
    const comment = feedbackPopup.querySelector('.robomate-discard-comment').value;

    // Store discard feedback
    if (!this.currentFeedback) {
      this.currentFeedback = { edits: [], deletionReasons: [] };
    }
    this.currentFeedback.discardReason = selectedReason;
    this.currentFeedback.discardComment = comment;

    feedbackPopup.remove();
    onComplete();
  };
}
```

### 5. Update captureFeedback() Method

```javascript
async captureFeedback(action) {
  const commentText = document.getElementById('robomate-feedback-comment')?.value || '';

  const feedback = {
    generationId: this.currentGenerationId || 'unknown',
    timestamp: Date.now(),
    action: action,
    comment: commentText,
    edits: this.currentFeedback.edits || [],
    deletionReasons: this.currentFeedback.deletionReasons || [],
    originalReply: this.currentReply || '',
    finalReply: this.getFinalReplyText(),

    // NEW: Add discard feedback if present
    discardReason: this.currentFeedback.discardReason || null,
    discardComment: this.currentFeedback.discardComment || null
  };

  // Send to backend
  await this.submitFeedback(feedback);
}
```

## CSS Styling

```css
/* Discard Feedback Popup */
.robomate-discard-feedback {
  position: absolute;
  top: 50%;
  left: 50%;
  transform: translate(-50%, -50%);
  z-index: 10001;
  background: white;
  border-radius: 12px;
  box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
  padding: 24px;
  max-width: 500px;
  width: 90%;
}

.robomate-discard-feedback-title {
  font-size: 18px;
  font-weight: 600;
  margin-bottom: 16px;
  color: #333;
}

.robomate-discard-feedback-options {
  display: flex;
  flex-wrap: wrap;
  gap: 8px;
  margin-bottom: 16px;
}

.robomate-discard-tag {
  padding: 8px 16px;
  border: 2px solid #ddd;
  border-radius: 8px;
  background: white;
  cursor: pointer;
  font-size: 14px;
  transition: all 0.2s;
}

.robomate-discard-tag:hover {
  border-color: #FFD700;
  background: rgba(255, 215, 0, 0.1);
}

.robomate-discard-tag.selected {
  border-color: #FFD700;
  background: #FFD700;
  color: #1a1a1a;
  font-weight: 600;
}

.robomate-discard-comment {
  width: 100%;
  padding: 10px;
  border: 2px solid #ddd;
  border-radius: 6px;
  font-size: 13px;
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  resize: vertical;
  margin-bottom: 16px;
}

.robomate-discard-comment:focus {
  outline: none;
  border-color: #FFD700;
}

.robomate-discard-actions {
  display: flex;
  justify-content: flex-end;
  gap: 12px;
}

.robomate-discard-skip {
  padding: 8px 16px;
  border: 2px solid #ddd;
  border-radius: 6px;
  background: white;
  cursor: pointer;
  font-size: 14px;
}

.robomate-discard-submit {
  padding: 8px 16px;
  border: none;
  border-radius: 6px;
  background: #FFD700;
  color: #1a1a1a;
  font-weight: 600;
  cursor: pointer;
  font-size: 14px;
}

.robomate-discard-submit:hover {
  background: #f0c800;
}
```

## Analytics Queries

### Most Common Discard Reasons

```sql
SELECT
  discard_reason,
  COUNT(*) as count,
  ROUND(100.0 * COUNT(*) / (SELECT COUNT(*) FROM feedback WHERE action = 'discarded'), 1) as percentage
FROM feedback
WHERE action = 'discarded'
  AND discard_reason IS NOT NULL
GROUP BY discard_reason
ORDER BY count DESC;
```

### Discard Reasons with Examples

```sql
SELECT
  f.discard_reason,
  f.discard_comment,
  g.ticket_url,
  g.customer_name,
  g.subject,
  substr(g.generated_reply, 1, 100) || '...' as reply_preview
FROM feedback f
JOIN generations g ON f.generation_id = g.id
WHERE f.action = 'discarded'
  AND f.discard_reason IS NOT NULL
ORDER BY f.created_at DESC
LIMIT 20;
```

### Discard Rate by User

```sql
SELECT
  u.name,
  COUNT(*) as total_generations,
  SUM(CASE WHEN f.action = 'discarded' THEN 1 ELSE 0 END) as discarded,
  SUM(CASE WHEN f.action = 'sent' THEN 1 ELSE 0 END) as sent,
  ROUND(100.0 * SUM(CASE WHEN f.action = 'discarded' THEN 1 ELSE 0 END) / COUNT(*), 1) as discard_rate
FROM users u
JOIN generations g ON u.id = g.user_id
LEFT JOIN feedback f ON g.id = f.generation_id
GROUP BY u.id
ORDER BY discard_rate DESC;
```

## Implementation Checklist

- [ ] Add `discardReason` and `discardComment` to feedback data structure
- [ ] Update database schema with new fields
- [ ] Create `showDiscardFeedback()` method
- [ ] Update close button handler to show popup
- [ ] Update Escape key handler to show popup
- [ ] Add CSS styling for discard popup
- [ ] Update `captureFeedback()` to include discard fields
- [ ] Update backend to accept new fields
- [ ] Test discard flow end-to-end
- [ ] Add analytics queries for discard reasons

## Benefits

✅ **Understand rejection patterns**: See why entire generations are discarded
✅ **Improve AI prompts**: Identify systemic issues (tone, relevance, etc.)
✅ **Separate concerns**: Distinguish between "bad generation" vs "just needed tweaks"
✅ **User intent tracking**: Know when users prefer writing from scratch
✅ **Quantifiable feedback**: Tag-based reasons enable analytics
